//----------------------------------------------------------------------------//
//- CafAnbieterUebersicht.pas ------------------------------------------------//
//----------------------------------------------------------------------------//

//- Beschreibung -------------------------------------------------------------//
// Dieses Frame stellt eine Listview-bersicht fr die Anbieter dar           //
//----------------------------------------------------------------------------//

unit CafAnbieterUebersicht;

//------------------------------------------------------------------------------

interface

//------------------------------------------------------------------------------

uses 
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ComCtrls, ActnList, CaoCheck, Menus, CaoPersonenListe;

//------------------------------------------------------------------------------

type
  TFrmAnbieterUebersicht = class(TFrame)
    m_LivAnbieterUebersicht : TListView;
    m_AclAnbieter           : TActionList;
    m_ActBearbeiten         : TAction;
    m_ActKopieren           : TAction;
    m_ActLoeschen           : TAction;
    m_ActNeu                : TAction;
    m_PopAnbieter           : TPopupMenu;
    m_MnuBearbeiten         : TMenuItem;
    m_MnuKopieren           : TMenuItem;
    m_MnuLoeschen           : TMenuItem;
    m_MnuNeu                : TMenuItem;

    //--------------------------------------------------------------------------

    procedure ActKopierenExecute              (Sender  : TObject);
    procedure ActBearbeitenExecute            (Sender  : TObject);
    procedure ActLoeschenExecute              (Sender  : TObject);
    procedure LivAnbieterUebersichtKeyDown    (Sender  : TObject;
                                               var Key : Word;
                                               Shift   : TShiftState);
    procedure LivAnbieterUebersichtColumnClick(Sender  : TObject;
                                               Column  : TListColumn);
    procedure ActNeuExecute                   (Sender  : TObject);
    procedure LivAnbieterUebersichtSelectItem(Sender: TObject;
                                            Item: TListItem; Selected: Boolean);
    
    //--------------------------------------------------------------------------

  private
    m_obCaoCheck          : TCaoCheck;
    m_obAnbieterListe     : TCaoPersonenListe;
    m_iSortierung         : Integer;
    m_CallingHandle       : THandle;

    //--------------------------------------------------------------------------

    procedure PrvLoadData;
    procedure PrvInitListView;
    function  PrvShowDlgAnbieter(obAnbieter: TCaoPersonen): Integer;
    function  PrvGetSelectedPerson: TCaoPersonen;

  //----------------------------------------------------------------------------

  public
    property  SelectedPerson: TCaoPersonen read PrvGetSelectedPerson;

    //--------------------------------------------------------------------------

    procedure SetFocusInListView(obPerson: TCaoPersonen);

    //--------------------------------------------------------------------------

    procedure CreateFrame       (obAnbieterListe: TCaoPersonenListe);
    procedure DestroyFrame;
    procedure ShowFrame         (CallingHandle  : THandle;
                                 obAnbieterListe: TCaoPersonenListe);
    function  IsAnbieterSelected: Boolean;

  //----------------------------------------------------------------------------

  end; // TFrmAnbieterUebersicht

//------------------------------------------------------------------------------

implementation

//------------------------------------------------------------------------------

{$R *.dfm}

//------------------------------------------------------------------------------

uses
  CadAnbieter, CamConst;

//------------------------------------------------------------------------------

const
  // Position der Eintrge in der ListView
  // Die Initialen werden im Listview unter "Caption" dargestellt
  c_iPosInPersonenListVorname   = 0;
  c_iPosInPersonenListName      = 1;
  c_iPosInPersonenListPlz       = 2;
  c_iPosInPersonenListOrt       = 3;

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.ActKopierenExecute(Sender: TObject);
(*
  ffnen des Anbieter-Dialogs zum Kopieren eines Anbieters
*)
var
  iTempModalResult: Integer;
  obAnbieter      : TCaoPersonen;
begin
  if m_obCaoCheck.ListViewItemIsSelected(m_LivAnbieterUebersicht) then begin
    // Kopieren des Selektierten Anbieters
    obAnbieter := m_obAnbieterListe.CopyAndAddPerson(TCaoPersonen
                                       (m_LivAnbieterUebersicht.Selected.Data));
    obAnbieter.iPersonenNummer := m_obAnbieterListe.GetMaxPersonenNr + 1;
    // Den neuen Eintrag als "Kopiert" markieren
    obAnbieter.iStatus := c_iModusCopy;
    // Anbieterdialog mit den Kopierten Datenstzen anzeigen
    iTempModalResult := PrvShowDlgAnbieter(obAnbieter);
    // Wurde beim Verlassen des Anbieter-Dialogs "Abbrechen" gewhlt, so muss
    // der eben erstellte Anbieter wieder gelscht werden
    if iTempModalResult = mrCancel then begin
      m_obAnbieterListe.DeletePerson(obAnbieter);
    end // if
    else begin
      m_obAnbieterListe.SaveToDb;
    end; // else
    PrvInitListView;
    SetFocusInListView(obAnbieter);
  end; // if 
end; // TFrmAnbieterUebersicht.ActKopierenExecute

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.ActBearbeitenExecute(Sender: TObject);
(*
  ffnen des Anbieter-Dialogs zum bearbeiten eines Anbieters
  -> Wird ein Anbieter editiert, so muss er neu angelegt werden, da sonst die
     nderungen nicht mit "Abbrechen" rckgngig gemacht werden knnten
*)
var
  obPerson        : TCaoPersonen;   
begin
  if m_obCaoCheck.ListViewItemIsSelected(m_LivAnbieterUebersicht) then begin
    obPerson := TCaoPersonen(m_LivAnbieterUebersicht.Selected.Data);
    // Der zu bearbeiteten Person den Status "Edit" zuweisen
    obPerson.iStatus := c_iModusEdit;
    // Anbieterdialog mit den zu bearbeitenden Anbieterdaten anzeigen
    PrvShowDlgAnbieter(obPerson);
    m_obAnbieterListe.SaveToDb;
    PrvInitListView;
    SetFocusInListView(obPerson);
  end; // if
end; // TFrmAnbieterUebersicht.ActBearbeitenExecute

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.ActLoeschenExecute(Sender: TObject);
(*
  Kennzeichnet eine Person als gelscht
*)
begin
  // berprfen, ob in der ListView eine Person selektiert wurde
  if m_obCaoCheck.ListViewItemIsSelected(m_LivAnbieterUebersicht) then begin
    TCaoPersonen(m_LivAnbieterUebersicht.Selected.Data).iStatus := c_iModusDelete;
    PrvInitListView;
  end; // if
end; // TFrmAnbieterUebersicht.ActLoeschenExecute

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.CreateFrame(obAnbieterListe: TCaoPersonenListe);
(*
  Da das Frame kein Ereignis "OnCreate" besitzt, wird es hier simuliert
  -> Muss im OnCreate des Owners aufgerufen werden
*)
begin
  m_obCaoCheck           := TCaoCheck.Create;
  m_obAnbieterListe      := obAnbieterListe;
end; // TFrmAnbieterUebersicht.CreateFrame

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.DestroyFrame;
(*
  Da das Frame kein Ereignis "OnDestroy" besitzt, wird es hier simuliert
  -> Muss im OnDestroy des Owners aufgerufen werden
*)
begin
  m_obCaoCheck.Free;
end; // TFrmAnbieterUebersicht.DestroyFrame

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.LivAnbieterUebersichtKeyDown(Sender : TObject;
                                                              var Key: Word;
                                                              Shift  : TShiftState);
begin
  // Wird auf der Tastatur "Delete" gedrckt, so soll der selektierte Eintrag
  // gelscht werden
  if Key = VK_Delete then begin
    ActLoeschenExecute(Sender);
  end; // if
end;

//------------------------------------------------------------------------------

function TFrmAnbieterUebersicht.IsAnbieterSelected: Boolean;
(*
  berprfen, ob in der ListView ein Anbieter selektiert wurde
*)
begin
  Result := m_LivAnbieterUebersicht.Selected <> nil;
end; // TFrmAnbieterUebersicht.IsAnbieterSelected

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.ShowFrame(CallingHandle: THandle;
                                           obAnbieterListe: TCaoPersonenListe);
(*
  Da das Frame kein Ereignis "OnShow" besitzt, wird es hier simuliert
  -> Muss im OnShow des Owners aufgerufen werden
*)
begin
  m_CallingHandle   := CallingHandle;
  m_obAnbieterListe := obAnbieterListe;
  Self.PrvInitListView;
  PrvLoadData;
end; // TFrmAnbieterUebersicht.ShowFrame

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.PrvLoadData;
begin
  PrvInitListView;
end; // TFrmAnbieterUebersicht.PrvLoadData

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.PrvInitListView;
(*
  Sortiert die AnbieterListe und fllt die Anbieter-Listview mit Daten ab
*)
var
  iIndex        : Integer;
  obListViewItem: TListItem;
begin
  m_LivAnbieterUebersicht.Clear;
  // Sortieren der AnbieterListe
  case m_iSortierung of
    c_iByInitialen : m_obAnbieterListe.SortByInitialen;
    c_iByVorname   : m_obAnbieterListe.SortByVorname;
    c_iByName      : m_obAnbieterListe.SortByName;
    c_iByPlz       : m_obAnbieterListe.SortByPlz;
    c_iByOrt       : m_obAnbieterListe.SortByOrt;
  end; // case
  // Abfllen der AnbieterListe in die Listview
  for iIndex := 0 to m_obAnbieterListe.Count - 1 do begin
    if m_obAnbieterListe.Items[iIndex].iStatus <> c_iModusDelete then begin
      obListViewItem         := m_LivAnbieterUebersicht.Items.Add;
      obListViewItem.Data    := m_obAnbieterListe      .Items[iIndex];
      obListViewItem.Caption := (m_obAnbieterListe     .Items[iIndex].sInitialen);
      obListViewItem.SubItems.Add(m_obAnbieterListe    .Items[iIndex].sVorname);
      obListViewItem.SubItems.Add(m_obAnbieterListe    .Items[iIndex].sName);
      obListViewItem.SubItems.Add(m_obAnbieterListe    .Items[iIndex].sPlz);
      obListViewItem.SubItems.Add(m_obAnbieterListe    .Items[iIndex].sOrt);
    end; // if
  end; // for
end; // TFrmAnbieterUebersicht.PrvInitListView

//------------------------------------------------------------------------------

function TFrmAnbieterUebersicht.PrvShowDlgAnbieter
                                 (obAnbieter: TCaoPersonen): Integer;
begin
// Solange wiederholen, bis Gltige Initialen eingegeben wurden, oder
// der Dialog mit "Abbrechen" verlassen wurde
  repeat
    Result := CaDlgAnbieter.ShowModal(obAnbieter);
    if (obAnbieter.iStatus = c_iModusEdit) and (Result <> mrCancel) then begin
      // Wurde der Dialog beim Editieren mit "OK" verlassen, so wird der
      // ursprngliche Eintrag als gelscht markiert
      TCaoPersonen(m_LivAnbieterUebersicht.Selected.Data).iStatus := c_iModusDelete;
    end; // if
  until (not m_obAnbieterListe.InitialenExists
        (obAnbieter, Result <> mrCancel)) or (Result = mrCancel);
  if Result = mrRetry then begin
    PrvInitListView;
  end; // if
end; // TFrmAnbieterUebersicht.PrvShowDlgAnbieter

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.LivAnbieterUebersichtColumnClick
                                                          (Sender: TObject;
                                                           Column: TListColumn);
(*
  Beim Klicken auf die Spaltentitel wird die ListView neu sortiert
*)
begin
  m_iSortierung := Column.Index;
  PrvInitListView;
end; // TFrmAnbieterUebersicht.LivAnbieterUebersichtColumnClick

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.ActNeuExecute(Sender: TObject);
(*
  Action zum erfassen von neuen Anbietern
*)
var
  iTempModalResult: Integer;
  obAnbieter      : TCaoPersonen;
begin
  // Solange "Weitere erfassen" gedrckt wird, wird der Anbieter-Dialog wieder
  // angezeigt
  repeat
    obAnbieter                 := m_obAnbieterListe.CreatePerson;
    obAnbieter.iStatus         := c_iModusNeu;
    obAnbieter.iPersonenNummer := m_obAnbieterListe.GetMaxPersonenNr + 1;

    // Solange wiederholen, bis Gltige Initialen eingegeben wurden, oder
    // der Dialog mit "Abbrechen" verlassen wurde
    repeat
      iTempModalResult := CaDlgAnbieter.ShowModal(obAnbieter);
    until (not m_obAnbieterListe.InitialenExists(obAnbieter,
                iTempModalResult <> mrCancel)) or (iTempModalResult = mrCancel);

    // Nach jedem Bettigen von "Weitere erfassen" die ListView neu aufbauen
    if iTempModalResult = mrRetry then begin
      PrvInitListView;
    end; // if

  until iTempModalResult <> mrRetry;

  // Wurde beim Verlassen des Anbieter-Dialogs "Abbrechen" gewhlt, so muss
  // der eben erstellte Anbieter wieder gelscht werden
  if iTempModalResult = mrCancel then begin
    m_obAnbieterListe.DeletePerson(obAnbieter);
  end // if
  // Andernfalls werden die nderungen in der Anbieterliste gespeichert
  else begin
    m_obAnbieterListe.SaveToDb;
  end; // else
  PrvInitListView;
  SetFocusInListView(obAnbieter);
end; // TFrmAnbieterUebersicht.ActNeuExecute

//------------------------------------------------------------------------------

function TFrmAnbieterUebersicht.PrvGetSelectedPerson: TCaoPersonen;
begin
  Result := nil;
  if IsAnbieterSelected then begin
    Result := TCaoPersonen(m_LivAnbieterUebersicht.Selected.Data);
  end; // if
end; // TFrmAnbieterUebersicht.PrvGetSelectedPerson

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.LivAnbieterUebersichtSelectItem(Sender: TObject;
                                                                 Item: TListItem;
                                                                 Selected: Boolean);
begin
    PostMessage(m_CallingHandle, wm_UpdateFrmAnbieterUebersicht, 0, 0);
end; // TFrmAnbieterUebersicht.LivAnbieterUebersichtSelectItem

//------------------------------------------------------------------------------

procedure TFrmAnbieterUebersicht.SetFocusInListView(obPerson: TCaoPersonen);
(*
  Setzt den Fokus in der Listview auf das Objekt, das per Parameter mitgegeben
  wurde
*)
var
  iIndex: Integer;
begin
  for iIndex := 0 to m_LivAnbieterUebersicht.Items.Count -1 do begin
    if m_LivAnbieterUebersicht.Items[iIndex].Data = Pointer(obPerson) then begin
      m_LivAnbieterUebersicht.SetFocus;
      m_LivAnbieterUebersicht.Items[iIndex].Selected := True;
      Break;
    end; // if
  end; // for
end; // TFrmAnbieterUebersicht.SetFocusInListView

//------------------------------------------------------------------------------

end. // CafAnbieterUebersicht
